# Frontend - Utils (Funções Utilitárias)

Funções utilitárias reutilizáveis para formatação, manipulação de dados e helpers diversos.

## Localização

```
src/utils/
├── formatUtils.ts      # Formatação de valores (moeda, pontos, etc.)
├── numberFormat.ts     # Formatação numérica
├── encantabraTone.ts   # Cores/tons para Encantabra
└── animations.ts       # Helpers de animação
```

## formatUtils.ts

Funções para formatação de valores monetários, pontos e outros formatos.

### formatBRL()

Formata valores em Real brasileiro (R$).

```typescript
import { formatBRL } from '@/utils/formatUtils'

formatBRL(1234.56)        // "R$ 1.234,56"
formatBRL('1234.56')      // "R$ 1.234,56"
formatBRL(null)           // "R$ 0,00"
formatBRL(undefined)      // "R$ 0,00"
```

### formatPoints()

Formata valores como pontos.

```typescript
import { formatPoints } from '@/utils/formatUtils'

formatPoints(1234)                    // "1.234"
formatPoints(1234, { withUnit: true }) // "1.234 pts"
formatPoints(null)                    // "0"
```

### formatPeso()

Formata valores de peso (2 casas decimais).

```typescript
import { formatPeso } from '@/utils/formatUtils'

formatPeso(1.5)    // "1,50"
formatPeso(0.123)  // "0,12"
formatPeso(null)   // "0"
```

### formatPercent()

Formata valores como percentual.

```typescript
import { formatPercent } from '@/utils/formatUtils'

formatPercent(85.5)        // "85,5%"
formatPercent(100)          // "100%"
formatPercent(null)         // "0%"
```

### formatDate()

Formata datas no formato brasileiro.

```typescript
import { formatDate } from '@/utils/formatUtils'

formatDate('2025-01-15')     // "15/01/2025"
formatDate(new Date())        // Data atual formatada
formatDate(null)              // ""
```

### formatDateTime()

Formata data e hora.

```typescript
import { formatDateTime } from '@/utils/formatUtils'

formatDateTime('2025-01-15T10:30:00') // "15/01/2025 10:30"
```

## numberFormat.ts

Formatação numérica usando `Intl.NumberFormat`.

### Exemplo de Uso

```typescript
import { formatNumber } from '@/utils/numberFormat'

// Formatação padrão brasileira
formatNumber(1234.56)  // "1.234,56"

// Com opções customizadas
formatNumber(1234.56, {
  minimumFractionDigits: 2,
  maximumFractionDigits: 2
})
```

## encantabraTone.ts

Funções para obter cores/tons do tema Encantabra baseado em status ou valores.

### getToneByStatus()

Retorna o tom (cor) baseado no status.

```typescript
import { getToneByStatus } from '@/utils/encantabraTone'

getToneByStatus('aprovado')  // "success"
getToneByStatus('pendente')  // "warning"
getToneByStatus('rejeitado') // "danger"
```

### getToneByValue()

Retorna o tom baseado em um valor numérico (ex: percentual de aprovação).

```typescript
import { getToneByValue } from '@/utils/encantabraTone'

getToneByValue(100)  // "success" (verde)
getToneByValue(75)   // "progress" (azul)
getToneByValue(50)   // "warning" (amarelo)
getToneByValue(25)   // "danger" (vermelho)
```

## animations.ts

Helpers para animações e transições.

### Exemplo de Uso

```typescript
import { fadeIn, slideUp } from '@/utils/animations'

// Aplicar animação fade in
fadeIn(element)

// Aplicar animação slide up
slideUp(element)
```

## Uso em Componentes

### Exemplo: Formatação de Valores

```vue
<script setup lang="ts">
import { formatBRL, formatPoints } from '@/utils/formatUtils'

const meta = 1500.50
const pontos = 1234

const metaFormatada = formatBRL(meta)      // "R$ 1.500,50"
const pontosFormatados = formatPoints(pontos, { withUnit: true }) // "1.234 pts"
</script>

<template>
  <div>
    <p>Meta: {{ metaFormatada }}</p>
    <p>Pontos: {{ pontosFormatados }}</p>
  </div>
</template>
```

### Exemplo: Cores Dinâmicas

```vue
<script setup lang="ts">
import { getToneByValue } from '@/utils/encantabraTone'
import { computed } from 'vue'

const atingimento = 85

const tone = computed(() => getToneByValue(atingimento))
// tone.value = "progress" (azul para 85%)
</script>

<template>
  <div :class="`bg-${tone}`">
    Atingimento: {{ atingimento }}%
  </div>
</template>
```

## Boas Práticas

### ✅ Use Utils para Formatação

```typescript
// ✅ CORRETO
import { formatBRL } from '@/utils/formatUtils'
const formatted = formatBRL(value)

// ❌ ERRADO
const formatted = `R$ ${value.toFixed(2)}` // Não considera locale
```

### ✅ Trate Valores Nulos

As funções de formatação já tratam `null` e `undefined`, retornando valores padrão:

```typescript
formatBRL(null)      // "R$ 0,00"
formatPoints(null)   // "0"
formatDate(null)     // ""
```

### ✅ Use Tipos Corretos

```typescript
// ✅ CORRETO
const value: number | null = getValue()
const formatted = formatBRL(value)

// ❌ ERRADO
const formatted = formatBRL(value as number) // Pode quebrar se null
```

## Extensões

Para adicionar novas funções de formatação:

1. Adicione a função em `formatUtils.ts`
2. Exporte a função
3. Documente o uso
4. Adicione testes se necessário

```typescript
// Exemplo: Nova função de formatação
export function formatCNPJ(cnpj: string | null | undefined): string {
  if (!cnpj) return ''
  // Lógica de formatação
  return cnpj.replace(/^(\d{2})(\d{3})(\d{3})(\d{4})(\d{2})$/, '$1.$2.$3/$4-$5')
}
```

---

**Anterior**: [Router e Config](./05-router-config.md)  
**Próximo**: [Types](./07-types.md)  
**Voltar**: [Estrutura](./01-estrutura.md)
