# Frontend - Componentes Vue

## Estrutura de um Componente

```vue
<script setup lang="ts">
// 1. Imports
import { ref, computed, onMounted } from 'vue'
import type { Produto } from '@/types'

// 2. Types/Interfaces
interface Props {
  produtos: Produto[]
  loading?: boolean
}

// 3. Props
const props = withDefaults(defineProps<Props>(), {
  loading: false
})

// 4. Emits
const emit = defineEmits<{
  (e: 'select', produto: Produto): void
  (e: 'refresh'): void
}>()

// 5. Composables
const { data } = useSomeComposable()

// 6. Estado local
const selectedId = ref<string | null>(null)

// 7. Computed
const produtosFiltrados = computed(() => {
  return props.produtos.filter(p => p.atingido)
})

// 8. Métodos
const handleSelect = (produto: Produto) => {
  selectedId.value = produto.id
  emit('select', produto)
}

// 9. Lifecycle
onMounted(() => {
  console.log('Componente montado')
})
</script>

<template>
  <div class="produtos-list">
    <div v-if="loading" class="loading">Carregando...</div>
    <div v-else>
      <div
        v-for="produto in produtosFiltrados"
        :key="produto.id"
        :class="{ selected: selectedId === produto.id }"
        @click="handleSelect(produto)"
      >
        {{ produto.indicador }}
      </div>
    </div>
  </div>
</template>

<style scoped>
.produtos-list {
  /* Estilos */
}

.selected {
  background-color: var(--brad-color-primary);
}
</style>
```

## Componentes Globais

- `Header.vue`: Cabeçalho da aplicação
- `Footer.vue`: Rodapé
- `Filters.vue`: Barra de filtros global
- `TabsNavigation.vue`: Navegação por abas
- `ChatWidget.vue`: Widget de chat (Agent)
- `Toast.vue`: Notificações toast

## Componentes POBJ

- `ResumoKPI.vue`: Cards de KPIs
- `ResumoLegacy.vue`: Visualização legada do resumo
- `ClassicSemesterView.vue`: Visualização clássica por semestre
- `ProdutosCards.vue`: Cards de produtos
- `DetailViewBar.vue`: Barra de detalhes
- `DetailColumnDesigner.vue`: Designer de colunas

## Componentes Utilitários

- `Button.vue`: Botão customizado
- `Icon.vue`: Componente de ícone
- `SelectInput.vue`: Input de seleção
- `SearchableDropdown.vue`: Dropdown pesquisável
- `DatePopover.vue`: Seletor de data
- `PeriodSelector.vue`: Seletor de período
- `EmptyState.vue`: Estado vazio
- `ErrorState.vue`: Estado de erro

---

**Próximo**: [Router e Config](./05-router-config.md)
