# Frontend - Composables

Composables são funções reutilizáveis que encapsulam lógica e estado usando Composition API.

## useResumoData.ts

```typescript
import { ref, computed, watch, nextTick } from 'vue'
import type { ResumoPayload, Period } from '../types'
import type { FilterState } from './useGlobalFilters'
import { getResumo, type ResumoFilters } from '../services/resumoService'
import { useGlobalFilters } from './useGlobalFilters'

// Estado global (singleton)
const resumoPayload = ref<ResumoPayload | null>(null)
const resumoLoading = ref(false)
const resumoError = ref<string | null>(null)
const lastFilters = ref<ResumoFilters | null>(null)

// Sanitiza valores de filtro
function sanitizeValue(value?: string | null): string | undefined {
  if (!value) return undefined
  const trimmed = value.trim()
  if (!trimmed) return undefined
  const lower = trimmed.toLowerCase()
  // Remove valores "todos/todas"
  if (lower === 'todos' || lower === 'todas') return undefined
  return trimmed
}

// Constrói filtros a partir do estado
function buildFiltersFromState(
  state?: FilterState, 
  period?: Period
): ResumoFilters {
  const filters: ResumoFilters = {}
  
  if (state) {
    const segmento = sanitizeValue(state.segmento)
    const diretoria = sanitizeValue(state.diretoria)
    const regional = sanitizeValue(state.gerencia)
    const agencia = sanitizeValue(state.agencia)
    const gerente = sanitizeValue(state.gerente)
    const familia = sanitizeValue(state.familia)
    const indicador = sanitizeValue(state.indicador)
    
    // Mapeia status do frontend para backend
    let status: string | undefined = undefined
    if (state.status && state.status !== 'todos') {
      if (state.status === 'atingidos') {
        status = '01'
      } else if (state.status === 'nao') {
        status = '02'
      }
    }
    
    // Adiciona apenas valores definidos
    if (segmento) filters.segmento = segmento
    if (diretoria) filters.diretoria = diretoria
    if (regional) filters.regional = regional
    if (agencia) filters.agencia = agencia
    if (gerente) filters.gerente = gerente
    if (familia) filters.familia = familia
    if (indicador) filters.indicador = indicador
    if (status) filters.status = status
  }
  
  // Adiciona período
  if (period?.start) {
    filters.dataInicio = period.start
  }
  if (period?.end) {
    filters.dataFim = period.end
  }
  
  return filters
}

// Busca dados do resumo
async function fetchResumo(filters: ResumoFilters): Promise<void> {
  // Previne múltiplas requisições simultâneas
  if (resumoLoading.value) {
    return
  }
  
  lastFilters.value = filters
  resumoLoading.value = true
  resumoError.value = null
  
  try {
    const data = await getResumo(filters)
    if (data) {
      resumoPayload.value = data
    } else {
      resumoError.value = 'Não foi possível carregar o resumo'
    }
  } catch (error) {
    console.error('Erro ao carregar resumo:', error)
    resumoError.value = error instanceof Error ? error.message : 'Erro desconhecido'
  } finally {
    resumoLoading.value = false
  }
}

// Composable principal
export function useResumoData(
  filterState: Ref<FilterState> | ComputedRef<FilterState>,
  period: Ref<Period> | ComputedRef<Period>
) {
  const { filterTrigger, filterState: globalFilterState, period: globalPeriod } = useGlobalFilters()
  
  // Watch automático em mudanças de filtros
  watch(
    filterTrigger,
    async () => {
      await nextTick() // Aguarda atualizações do DOM
      const filters = buildFiltersFromState(globalFilterState.value, globalPeriod.value)
      await fetchResumo(filters)
    },
    { immediate: true } // Executa na inicialização
  )
  
  return {
    // Computed properties reativas
    resumo: computed(() => resumoPayload.value),
    produtos: computed(() => resumoPayload.value?.cards ?? []),
    produtosMensais: computed(() => resumoPayload.value?.classifiedCards ?? []),
    variavel: computed(() => resumoPayload.value?.variableCard ?? []),
    businessSnapshot: computed(() => resumoPayload.value?.businessSnapshot ?? emptySnapshot),
    loading: computed(() => resumoLoading.value),
    error: computed(() => resumoError.value),
    
    // Métodos
    loadResumo: async () => {
      const filters = buildFiltersFromState(filterState.value, period.value)
      await fetchResumo(filters)
    },
    clearData: () => {
      resumoPayload.value = null
      resumoError.value = null
      lastFilters.value = null
    }
  }
}
```

## useGlobalFilters.ts

Gerencia filtros globais da aplicação:

```typescript
import { ref, computed, watch } from 'vue'
import type { FilterState } from './useGlobalFilters'

const filterState = ref<FilterState>({
  segmento: null,
  diretoria: null,
  regional: null,
  agencia: null,
  gerente: null,
  familia: null,
  indicador: null,
  status: 'todos'
})

const period = ref<Period>({
  start: null,
  end: null
})

// Trigger para watch - incrementa a cada mudança
const filterTrigger = ref(0)

export function useGlobalFilters() {
  // Carrega do localStorage na inicialização
  const loadFromStorage = () => {
    const saved = localStorage.getItem('filters')
    if (saved) {
      try {
        filterState.value = { ...filterState.value, ...JSON.parse(saved) }
      } catch (e) {
        console.error('Erro ao carregar filtros:', e)
      }
    }
  }
  
  // Salva no localStorage
  const saveToStorage = () => {
    localStorage.setItem('filters', JSON.stringify(filterState.value))
  }
  
  // Watch que salva automaticamente
  watch(filterState, saveToStorage, { deep: true })
  
  // Atualiza filtro e dispara trigger
  const updateFilter = (key: keyof FilterState, value: any) => {
    filterState.value[key] = value
    filterTrigger.value++ // Dispara watchers
  }
  
  // Limpa todos os filtros
  const clearFilters = () => {
    filterState.value = {
      segmento: null,
      diretoria: null,
      // ... outros
    }
    filterTrigger.value++
  }
  
  return {
    filterState: computed(() => filterState.value),
    period: computed(() => period.value),
    filterTrigger: computed(() => filterTrigger.value),
    updateFilter,
    clearFilters,
    loadFromStorage
  }
}
```

## Composables Disponíveis

**Gestão de Estado:**
- `useGlobalFilters.ts`: Filtros globais da aplicação
- `useHierarchyFilters.ts`: Filtros hierárquicos
- `useOmegaFilters.ts`: Filtros específicos do Omega
- `usePeriodManager.ts`: Gerenciamento de períodos

**Dados:**
- `useResumoData.ts`: Dados do resumo
- `useDetalhesData.ts`: Dados de detalhes
- `useProdutos.ts`: Dados de produtos
- `useVariavel.ts`: Dados de variáveis
- `useInitCache.ts`: Cache de dados de inicialização
- `useCalendarioCache.ts`: Cache de calendário

**Funcionalidades:**
- `useOmega.ts`: Lógica do módulo Omega
- `useOmegaBulk.ts`: Operações em lote do Omega
- `useOmegaNotifications.ts`: Notificações Omega
- `usePDFExport.ts`: Exportação para PDF
- `useSelectSearch.ts`: Busca em seleções
- `useFeedback.ts`: Sistema de feedback
- `useAnimations.ts`: Animações
- `useBusinessDays.ts`: Cálculo de dias úteis

---

**Próximo**: [Componentes Vue](./04-componentes.md)
