# Frontend - Services (API Client)

## api.ts - Cliente HTTP Base

```typescript
import { API_BASE_URL, getApiKey } from '../config/api'
import type { ApiResponse } from '../types'

function buildUrl(path: string, params?: Record<string, any>): string {
  const cleanPath = path.startsWith('/') ? path : `/${path}`
  const base = API_BASE_URL.replace(/\/$/, '')
  const url = new URL(base + cleanPath)
  
  if (params) {
    Object.entries(params).forEach(([key, value]) => {
      if (value !== undefined && value !== null) {
        url.searchParams.append(key, String(value))
      }
    })
  }
  
  return url.toString()
}

function buildHeaders(customHeaders?: Record<string, string>): Record<string, string> {
  const headers: Record<string, string> = { ...customHeaders }
  
  // Adiciona API Key
  const apiKey = getApiKey()
  if (apiKey) {
    headers['X-API-Key'] = apiKey
  }
  
  // Adiciona funcional do usuário (se disponível)
  const userFuncional = localStorage.getItem('userFuncional') || 
                       sessionStorage.getItem('userFuncional')
  if (userFuncional) {
    headers['X-User-Funcional'] = userFuncional
  }
  
  return headers
}

export async function apiGet<T>(
  path: string,
  params?: Record<string, any>
): Promise<ApiResponse<T>> {
  try {
    const url = buildUrl(path, params)
    
    const response = await fetch(url, {
      method: 'GET',
      headers: buildHeaders(),
      cache: 'no-store' // Sempre busca dados frescos
    })
    
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}`)
    }
    
    const data = await response.json()
    
    // Normaliza resposta (pode vir com ou sem 'success')
    return data.success !== undefined
      ? data
      : { success: true, data }
  } catch (e) {
    return {
      success: false,
      error: e instanceof Error ? e.message : 'Unknown error'
    }
  }
}

// Similar para apiPost, apiPut, apiDelete...
```

## resumoService.ts - Serviço Específico

```typescript
import { apiGet } from './api'
import { ApiRoutes } from '../constants/apiRoutes'
import type { ResumoPayload, ResumoFilters } from '../types'

export async function getResumo(
  filters?: ResumoFilters
): Promise<ResumoPayload | null> {
  const params: Record<string, string> = {}
  
  // Constrói query params apenas com valores definidos
  if (filters) {
    if (filters.segmento) params.segmento = filters.segmento
    if (filters.diretoria) params.diretoria = filters.diretoria
    if (filters.regional) params.regional = filters.regional
    if (filters.agencia) params.agencia = filters.agencia
    if (filters.gerente) params.gerente = filters.gerente
    if (filters.gerenteGestao) params.gerenteGestao = filters.gerenteGestao
    if (filters.familia) params.familia = filters.familia
    if (filters.indicador) params.indicador = filters.indicador
    if (filters.subindicador) params.subindicador = filters.subindicador
    if (filters.dataInicio) params.dataInicio = filters.dataInicio
    if (filters.dataFim) params.dataFim = filters.dataFim
    if (filters.status) params.status = filters.status
  }
  
  const response = await apiGet<ResumoPayload>(ApiRoutes.RESUMO, params)
  
  if (response.success && response.data) {
    return response.data
  }
  
  console.error('Erro ao buscar resumo:', response.error)
  return null
}
```

## Serviços Disponíveis

- `api.ts`: Cliente HTTP base
- `resumoService.ts`: Serviço de resumo
- `detalhesService.ts`: Serviço de detalhes
- `rankingService.ts`: Serviço de ranking
- `produtosService.ts`: Serviço de produtos
- `execService.ts`: Serviço de visão executiva
- `simuladorService.ts`: Serviço de simulador
- `variavelService.ts`: Serviço de variáveis
- `calendarioService.ts`: Serviço de calendário
- `initService.ts`: Serviço de inicialização
- `omegaService.ts`: Serviço Omega
- `encantabraService.ts`: Serviço Encantabra
- `agentService.ts`: Serviço do chatbot

---

**Próximo**: [Composables](./03-composables.md)
