# Deploy - Configuração no XAMPP

> Tutorial completo para rodar o projeto localmente usando XAMPP, servidor PHP embutido e build de produção

## Pré-requisitos

- XAMPP instalado e configurado
- Node.js instalado (versão 20.19.0 ou superior)
- PHP 8.0+ disponível no PATH
- Composer instalado (para backend)

## Passo 1: Obter o IP da Máquina

### Windows

1. Abra o **Prompt de Comando** ou **PowerShell**
2. Execute:
   ```cmd
   ipconfig
   ```
3. Procure por **IPv4 Address** na seção da sua placa de rede (ex: `192.168.1.100`)

### Linux/Mac

```bash
ip addr show  # Linux
ifconfig      # Mac
```

**Anote o IP encontrado** (ex: `192.168.1.100`)

---

## Passo 2: Gerar o Build do Frontend

1. Abra o terminal na pasta do projeto
2. Navegue até a pasta `frontend`:
   ```bash
   cd frontend
   ```
3. Instale as dependências (se ainda não instalou):
   ```bash
   npm install
   ```
4. Gere o build de produção:
   ```bash
   npm run build
   ```
5. Aguarde a conclusão. Os arquivos serão gerados em `frontend/dist/`

**Verificação:** Confirme que a pasta `frontend/dist/` contém:
- `index.html`
- Pasta `assets/` com arquivos JS e CSS

---

## Passo 3: Alterar o IP no arquivo JavaScript compilado

O frontend precisa saber qual é o IP do backend. Após o build, o endereço `localhost:8081` fica hardcoded no arquivo JavaScript. Precisamos alterá-lo manualmente.

1. Abra a pasta `frontend/dist/assets/`
2. Localize o arquivo principal JavaScript (geralmente nomeado como `index.XXXXX.js`, onde `XXXXX` é um hash gerado pelo build)
   - Exemplo: `index.BIXVB7E_.js` ou `index.abc123.js`
3. Abra este arquivo em um editor de texto (Notepad++, VS Code, etc.)
4. Use a função de **Buscar e Substituir** (Ctrl+H) para encontrar:
   ```
   localhost:8081
   ```
5. Substitua **todas as ocorrências** por:
   ```
   SEU_IP:8081
   ```
   Onde `SEU_IP` é o IP obtido no Passo 1.

**Exemplo** (com IP `192.168.1.100`):
- **Buscar:** `localhost:8081`
- **Substituir por:** `192.168.1.100:8081`

**⚠️ Importante:**
- Use a função "Substituir Tudo" para garantir que todas as ocorrências sejam alteradas
- O nome do arquivo muda a cada build (por causa do hash), então sempre verifique qual é o arquivo principal
- Use o **mesmo IP** que você obteu no Passo 1
- A porta `8081` deve ser a mesma que será usada no servidor PHP embutido (próximo passo)
- Salve o arquivo após a alteração

**Dica:** Se houver múltiplos arquivos JavaScript em `assets/`, procure pelo arquivo que contém a string `localhost:8081`. Geralmente é o arquivo `index.*.js` principal.

---

## Passo 4: Configurar o Backend (Servidor PHP Embutido)

O backend Symfony precisa rodar em um servidor PHP. Vamos usar o servidor embutido do PHP.

1. Abra um **novo terminal** (mantenha o anterior aberto)
2. Navegue até a pasta `backend`:
   ```bash
   cd backend
   ```
3. Verifique se o arquivo `.env` está configurado:
   ```bash
   # Se não existir, copie o .env.example
   cp .env.example .env
   ```
4. Edite o `.env` e configure:
   ```env
   DATABASE_URL=mysql://root:@localhost:3306/nome_do_banco?serverVersion=8.0
   API_KEY=sua-chave-secreta-aqui
   ```
5. Instale as dependências (se ainda não instalou):
   ```bash
   composer install
   ```
6. Inicie o servidor PHP embutido na porta 8081:
   ```bash
   php -S SEU_IP:8081 -t public
   ```
   
   **Exemplo** (com IP `192.168.1.100`):
   ```bash
   php -S 192.168.1.100:8081 -t public
   ```

**⚠️ Importante:**
- Use o **mesmo IP** obtido no Passo 1
- A porta `8081` deve ser a mesma configurada no `index.html`
- **Mantenha este terminal aberto** enquanto usar a aplicação

**Verificação:** Acesse no navegador:
```
http://SEU_IP:8081/api/doc
```
Você deve ver a documentação Swagger da API.

---

## Passo 5: Configurar o XAMPP para Servir o Frontend

### Opção A: Usar htdocs do XAMPP (Recomendado)

1. Abra a pasta `htdocs` do XAMPP (geralmente em `C:\xampp\htdocs\`)
2. Crie uma pasta para o projeto (ex: `pobj`)
3. Copie **todo o conteúdo** da pasta `frontend/dist/` para `C:\xampp\htdocs\pobj\`
4. Inicie o Apache no XAMPP Control Panel
5. Acesse no navegador:
   ```
   http://localhost/pobj
   ```

### Opção B: Usar Virtual Host (Avançado)

1. Edite o arquivo `C:\xampp\apache\conf\extra\httpd-vhosts.conf`
2. Adicione:

```apache
<VirtualHost *:80>
    ServerName pobj.local
    DocumentRoot "C:/caminho/do/projeto/frontend/dist"
    
    <Directory "C:/caminho/do/projeto/frontend/dist">
        Options Indexes FollowSymLinks
        AllowOverride All
        Require all granted
        
        # Suporte para Vue Router (SPA)
        RewriteEngine On
        RewriteBase /
        RewriteRule ^index\.html$ - [L]
        RewriteCond %{REQUEST_FILENAME} !-f
        RewriteCond %{REQUEST_FILENAME} !-d
        RewriteRule . /index.html [L]
    </Directory>
</VirtualHost>
```

3. Edite o arquivo `C:\Windows\System32\drivers\etc\hosts` (como administrador):
   ```
   127.0.0.1    pobj.local
   ```
4. Reinicie o Apache no XAMPP
5. Acesse: `http://pobj.local`

---

## Passo 6: Verificar a Configuração

### Checklist

- [ ] Build do frontend gerado em `frontend/dist/`
- [ ] IP alterado no arquivo JavaScript em `frontend/dist/assets/index.*.js`
- [ ] Backend rodando em `http://SEU_IP:8081`
- [ ] Frontend acessível via XAMPP
- [ ] API_KEY configurada no backend

### Teste Completo

1. Abra o navegador
2. Acesse o frontend (via XAMPP)
3. Abra o **Console do Desenvolvedor** (F12)
4. Vá para a aba **Network**
5. Tente fazer login ou carregar dados
6. Verifique se as requisições estão indo para `http://SEU_IP:8081/api/...`

**Se as requisições falharem:**
- Verifique se o IP está correto no arquivo JavaScript (`assets/index.*.js`)
- Verifique se todas as ocorrências de `localhost:8081` foram substituídas
- Verifique se o backend está rodando na porta 8081
- Verifique se o firewall não está bloqueando a porta 8081
- Verifique se a `API_KEY` está correta

---

## Solução de Problemas

### Erro: "Cannot GET /"

**Causa:** Vue Router não está configurado no servidor.

**Solução:** Adicione um arquivo `.htaccess` em `frontend/dist/`:

```apache
<IfModule mod_rewrite.c>
  RewriteEngine On
  RewriteBase /
  RewriteRule ^index\.html$ - [L]
  RewriteCond %{REQUEST_FILENAME} !-f
  RewriteCond %{REQUEST_FILENAME} !-d
  RewriteRule . /index.html [L]
</IfModule>
```

### Erro: CORS no navegador

**Causa:** Backend não está permitindo requisições do frontend.

**Solução:** Verifique se o `CorsPreflightSubscriber` está configurado no backend para aceitar requisições do IP do frontend.

### Erro: "Connection refused" ou timeout

**Causa:** Firewall bloqueando a porta 8081 ou IP incorreto no arquivo JavaScript.

**Solução:**
1. Verifique se o IP está correto no arquivo JavaScript (`assets/index.*.js`)
2. Verifique se todas as ocorrências de `localhost:8081` foram substituídas
3. Adicione exceção no firewall do Windows para a porta 8081
4. Verifique se o servidor PHP está realmente rodando

### Erro: Assets não carregam (404)

**Causa:** Caminhos relativos incorretos.

**Solução:** Verifique se o `base` no `vite.config.ts` está como `/` e se os arquivos estão na pasta correta do XAMPP.

---

## Estrutura Final

```
Projeto/
├── backend/
│   ├── public/          # Servido por php -S na porta 8081
│   └── .env            # Configurado com DATABASE_URL e API_KEY
│
└── frontend/
    └── dist/           # Copiado para C:\xampp\htdocs\pobj\
        ├── index.html
        └── assets/      # Arquivos JS e CSS
            └── index.*.js  # Arquivo JavaScript com IP alterado
```

---

## Comandos Rápidos

### Iniciar Backend
```bash
cd backend
php -S 192.168.1.100:8081 -t public
```

### Gerar Build do Frontend
```bash
cd frontend
npm run build
```

### Atualizar Frontend no XAMPP
```bash
# Copiar dist para htdocs
xcopy /E /I /Y frontend\dist\* C:\xampp\htdocs\pobj\
```

---

## Dicas

1. **Desenvolvimento:** Use `npm run dev` no frontend para desenvolvimento (não precisa do XAMPP)
2. **Produção Local:** Use o build + XAMPP para simular ambiente de produção
3. **IP Dinâmico:** Se seu IP mudar, você precisará:
   - Regenerar o build (`npm run build`)
   - Alterar o IP novamente no arquivo `assets/index.*.js`
   - Reiniciar o servidor PHP
4. **Porta Alternativa:** Se a porta 8081 estiver ocupada, use outra (ex: 8082) e atualize o IP no arquivo JavaScript
5. **Busca Rápida:** Use `Ctrl+F` no arquivo JavaScript para encontrar rapidamente `localhost:8081` e substituir

---

**Anterior**: [Build de Produção](./02-build.md)  
**Voltar**: [Variáveis de Ambiente](./01-variaveis-ambiente.md)
