# Banco de Dados - Relacionamentos e Índices

## Diagrama de Relacionamentos

### Hierarquia Organizacional

```
segmentos (1)
  └── diretorias (N)
        └── regionais (N)
              └── agencias (N)
                    └── d_estrutura (N)
                          └── cargos (1)
```

**Relacionamentos:**
- `diretorias.segmento_id` → `segmentos.id`
- `regionais.diretoria_id` → `diretorias.id`
- `agencias.regional_id` → `regionais.id`
- `d_estrutura.segmento_id` → `segmentos.id` (opcional)
- `d_estrutura.diretoria_id` → `diretorias.id` (opcional)
- `d_estrutura.regional_id` → `regionais.id` (opcional)
- `d_estrutura.agencia_id` → `agencias.id` (opcional)
- `d_estrutura.cargo_id` → `cargos.id`

---

### Hierarquia de Produtos

```
familia (1)
  └── indicador (N)
        └── subindicador (N)
              └── d_produtos (N)
```

**Relacionamentos:**
- `indicador.familia_id` → `familia.id` (opcional)
- `subindicador.indicador_id` → `indicador.id`
- `d_produtos.familia_id` → `familia.id`
- `d_produtos.indicador_id` → `indicador.id`
- `d_produtos.subindicador_id` → `subindicador.id` (opcional)

---

### Tabelas Fato → Dimensão

```
d_calendario (1)
  └── f_meta (N) [via data_meta]
  └── f_realizados (N) [via data_realizado]
  └── f_pontos (N) [via data_realizado]
  └── f_variavel (N) [via dt_atualizacao]
  └── f_detalhes (N) [via dt_cadastro, competencia]
  └── f_campanhas (N) [via data]
  └── f_leads_propensos (N) [via database, data_contato]
  └── f_historico_ranking_pobj (N) [via data]

d_estrutura (1)
  └── f_meta (N) [via funcional]
  └── f_realizados (N) [via funcional]
  └── f_pontos (N) [via funcional]
  └── f_variavel (N) [via funcional]
  └── f_detalhes (N) [via funcional]
  └── f_historico_ranking_pobj (N) [via funcional]

d_produtos (1)
  └── f_meta (N) [via produto_id]
  └── f_realizados (N) [via produto_id]
  └── f_pontos (N) [via produto_id]
  └── f_detalhes (N) [via id_produto]

d_status_indicadores (1)
  └── f_detalhes (N) [via status_id]
```

---

### Sistema Omega

```
omega_departamentos (1)
  └── omega_status (N) [via departamento_id]
  └── omega_chamados (N) [via team_id]

omega_usuarios (1)
  └── omega_chamados (N) [via requester_id]
  └── omega_chamados (N) [via owner_id]

omega_status (1)
  └── omega_chamados (N) [via status]
```

---

## Índices Estratégicos

### Índices de Performance para Queries de Agregação

#### Tabelas Fato

**`f_meta`:**
- `ix_meta_data`: Otimiza filtros por data
- `ix_meta_func_data`: Otimiza queries por funcionário e data (composite)
- `fk_f_meta__produto`: Otimiza joins com produtos

**`f_realizados`:**
- `idx_fr_data`: Otimiza filtros por data
- `idx_fr_func_data`: Otimiza queries por funcionário e data (composite)
- `idx_fr_contrato`: Otimiza busca por contrato
- `idx_fr_produto`: Otimiza joins com produtos

**`f_pontos`:**
- `idx_fp_funcional`: Otimiza filtros por funcionário
- `idx_fp_produto`: Otimiza joins com produtos
- `idx_fp_data_realizado`: Otimiza filtros por data

**`f_detalhes`:**
- `idx_fd_funcional`: Otimiza filtros por funcionário
- `idx_fd_registro`: Otimiza busca por registro
- `idx_fd_produto`: Otimiza joins com produtos
- `idx_fd_dt_cadastro`: Otimiza filtros por data de cadastro
- `idx_fd_competencia`: Otimiza filtros por competência

**`f_campanhas`:**
- `idx_f_campanhas_data`: Otimiza filtros por data
- `idx_f_campanhas_diretoria`: Otimiza filtros por diretoria
- `idx_f_campanhas_gerencia`: Otimiza filtros por gerência
- `idx_f_campanhas_indicador`: Otimiza filtros por indicador
- `idx_f_campanhas_unidade`: Otimiza filtros hierárquicos (composite)

**`f_leads_propensos`:**
- `idx_f_leads_calendario`: Otimiza filtros por data base
- `idx_f_leads_contato`: Otimiza filtros por data de contato
- `idx_f_leads_unidade`: Otimiza filtros hierárquicos (composite)
- `idx_f_leads_produto`: Otimiza filtros por produto (composite)
- `idx_f_leads_diretoria`: Otimiza filtros por diretoria
- `idx_f_leads_regional`: Otimiza filtros por regional

**`f_historico_ranking_pobj`:**
- `idx_hist_data`: Otimiza filtros por data
- `idx_hist_funcional`: Otimiza filtros por funcionário
- `idx_hist_ranking`: Otimiza ordenação por ranking

---

### Índices para Filtros Hierárquicos

**`d_estrutura`:**
- `fk_estrutura_cargo`: Otimiza filtros por cargo
- `fk_estrutura_segmento`: Otimiza filtros por segmento
- `fk_estrutura_diretoria`: Otimiza filtros por diretoria
- `fk_estrutura_regional`: Otimiza filtros por regional
- `fk_estrutura_agencia`: Otimiza filtros por agência

**`d_calendario`:**
- `idx_d_calendario_mes`: Otimiza filtros por ano e mês (composite)

**`d_produtos`:**
- `idx_familia`: Otimiza filtros por família
- `idx_indicador`: Otimiza filtros por indicador
- `idx_subindicador`: Otimiza filtros por subindicador

---

### Índices para Sistema Omega

**`omega_chamados`:**
- `idx_omega_chamados_status`: Otimiza filtros por status
- `idx_omega_chamados_team`: Otimiza filtros por time/departamento
- `idx_omega_chamados_requester`: Otimiza filtros por solicitante
- `idx_omega_chamados_owner`: Otimiza filtros por responsável

**`omega_status`:**
- `idx_omega_status_departamento`: Otimiza filtros por departamento

---

## Constraints e Integridade Referencial

### Foreign Keys com ON UPDATE CASCADE

As seguintes foreign keys usam `ON UPDATE CASCADE` para manter consistência:

- `f_realizados.data_realizado` → `d_calendario.data`
- `f_pontos.data_realizado` → `d_calendario.data`
- `f_variavel.dt_atualizacao` → `d_calendario.data`
- `f_detalhes.dt_cadastro` → `d_calendario.data`
- `f_detalhes.competencia` → `d_calendario.data`
- `f_historico_ranking_pobj.data` → `d_calendario.data`
- `f_realizados.funcional` → `d_estrutura.funcional`
- `f_pontos.funcional` → `d_estrutura.funcional`
- `f_variavel.funcional` → `d_estrutura.funcional`
- `f_detalhes.funcional` → `d_estrutura.funcional`
- `f_historico_ranking_pobj.funcional` → `d_estrutura.funcional`
- `f_realizados.produto_id` → `d_produtos.id`
- `f_pontos.produto_id` → `d_produtos.id`
- `f_detalhes.id_produto` → `d_produtos.id`
- `f_detalhes.status_id` → `d_status_indicadores.id`

### Foreign Keys com ON DELETE

**ON DELETE CASCADE:**
- `f_leads_propensos.database` → `d_calendario.data` (se data for deletada, leads são deletados)

**ON DELETE SET NULL:**
- `f_leads_propensos.data_contato` → `d_calendario.data` (se data for deletada, data_contato vira NULL)
- `omega_chamados.owner_id` → `omega_usuarios.id` (se usuário for deletado, owner vira NULL)
- `omega_chamados.requester_id` → `omega_usuarios.id` (se usuário for deletado, requester vira NULL)
- `omega_chamados.status` → `omega_status.id` (se status for deletado, status vira NULL)
- `omega_chamados.team_id` → `omega_departamentos.departamento_id` (se departamento for deletado, team vira NULL)

---

## Otimizações Recomendadas

### Índices Compostos Adicionais (Sugestões)

Para melhorar performance em queries frequentes:

```sql
-- Agregações por funcionário e produto em período
CREATE INDEX idx_f_meta_func_prod_data ON f_meta(funcional, produto_id, data_meta);

-- Agregações por produto e data
CREATE INDEX idx_f_realizados_prod_data ON f_realizados(produto_id, data_realizado);

-- Filtros hierárquicos completos
CREATE INDEX idx_estrutura_hierarquia ON d_estrutura(segmento_id, diretoria_id, regional_id, agencia_id);

-- Filtros por cargo e estrutura
CREATE INDEX idx_estrutura_cargo_hierarquia ON d_estrutura(cargo_id, segmento_id, diretoria_id);
```

### Particionamento (Para Tabelas Grandes)

Para tabelas fato com muitos registros históricos, considere particionamento por data:

```sql
-- Exemplo: Particionar f_meta por ano
ALTER TABLE f_meta
PARTITION BY RANGE (YEAR(data_meta)) (
    PARTITION p2023 VALUES LESS THAN (2024),
    PARTITION p2024 VALUES LESS THAN (2025),
    PARTITION p2025 VALUES LESS THAN (2026),
    PARTITION p_future VALUES LESS THAN MAXVALUE
);
```

---

## Padrões de Query Otimizados

### Queries de Resumo (Agregação)

```sql
-- Resumo por funcionário e produto
SELECT 
    e.funcional,
    e.nome,
    p.id as produto_id,
    SUM(m.meta_mensal) as total_meta,
    SUM(r.realizado) as total_realizado
FROM f_meta m
INNER JOIN d_estrutura e ON m.funcional = e.funcional
INNER JOIN d_produtos p ON m.produto_id = p.id
LEFT JOIN f_realizados r ON r.funcional = m.funcional 
    AND r.produto_id = m.produto_id 
    AND r.data_realizado = m.data_meta
WHERE m.data_meta BETWEEN :data_inicio AND :data_fim
GROUP BY e.funcional, p.id;
```

### Queries Hierárquicas

```sql
-- Buscar funcionários por hierarquia
SELECT e.*
FROM d_estrutura e
WHERE e.segmento_id = :segmento_id
    AND (:diretoria_id IS NULL OR e.diretoria_id = :diretoria_id)
    AND (:regional_id IS NULL OR e.regional_id = :regional_id)
    AND (:agencia_id IS NULL OR e.agencia_id = :agencia_id);
```

### Queries com Calendário

```sql
-- Buscar dados por período com informações de calendário
SELECT 
    c.data,
    c.mes_nome,
    c.trimestre,
    SUM(f.meta_mensal) as total_meta
FROM f_meta f
INNER JOIN d_calendario c ON f.data_meta = c.data
WHERE c.ano = :ano
    AND c.mes = :mes
GROUP BY c.data, c.mes_nome, c.trimestre;
```

---

## Manutenção de Índices

### Verificar Fragmentação

```sql
-- MySQL/MariaDB
SELECT 
    TABLE_NAME,
    INDEX_NAME,
    CARDINALITY,
    INDEX_TYPE
FROM INFORMATION_SCHEMA.STATISTICS
WHERE TABLE_SCHEMA = 'nome_do_banco'
ORDER BY TABLE_NAME, INDEX_NAME;
```

### Reconstruir Índices

```sql
-- Reconstruir índice específico
ALTER TABLE f_meta DROP INDEX ix_meta_func_data;
ALTER TABLE f_meta ADD INDEX ix_meta_func_data (funcional, data_meta);

-- Otimizar tabela completa
OPTIMIZE TABLE f_meta;
```

---

**Anterior**: [Tabelas e Estrutura](./02-tabelas.md)  
**Voltar**: [Modelo de Dados](./01-modelo.md)
