# Backend - Specification Pattern

O projeto utiliza o **Specification Pattern** para construir queries dinâmicas de forma flexível e reutilizável.

## Conceito

O Specification Pattern permite combinar múltiplas condições de filtro usando operadores lógicos (AND, OR, NOT) de forma composável.

## Estrutura

```
Repository/Specification/
├── SpecificationInterface.php    # Interface base
├── CompositeSpecification.php     # Classe abstrata com operadores
├── FilterSpecification.php        # Implementação de filtros
├── AndSpecification.php           # Operador AND
├── OrSpecification.php             # Operador OR
└── NotSpecification.php            # Operador NOT
```

## Interface Base

```php
interface SpecificationInterface
{
    public function apply(QueryBuilder $qb, string $alias = 'e'): QueryBuilder;
    public function isSatisfiedBy(): bool;
}
```

## Uso Básico

### Criando uma Specification

```php
use App\Repository\Specification\FilterSpecification;
use App\Domain\DTO\FilterDTO;

// Criar specification com filtros
$filters = new FilterDTO();
$filters->setGerente('G123456');
$filters->setSegmento('1');

$spec = new FilterSpecification($filters);
```

### Combinando Specifications

```php
use App\Repository\Specification\FilterSpecification;
use App\Repository\Specification\AndSpecification;

$spec1 = new FilterSpecification($filters1);
$spec2 = new FilterSpecification($filters2);

// Combinar com AND
$combined = $spec1->and($spec2);

// Combinar com OR
$combined = $spec1->or($spec2);

// Negar
$negated = $spec1->not();
```

### Aplicando em QueryBuilder

```php
$qb = $entityManager->createQueryBuilder();
$qb->select('e')
   ->from(DEstrutura::class, 'e');

// Aplicar specification
$spec = new FilterSpecification($filters);
$qb = $spec->apply($qb, 'e');

$results = $qb->getQuery()->getResult();
```

## FilterSpecification

A `FilterSpecification` é a implementação principal que aplica filtros hierárquicos:

```php
class FilterSpecification extends CompositeSpecification
{
    private $filters;
    
    public function __construct(?FilterDTO $filters = null)
    {
        $this->filters = $filters;
    }
    
    public function apply(QueryBuilder $qb, string $alias = 'e'): QueryBuilder
    {
        // Aplica filtros hierárquicos
        // Prioridade: Gerente > Gerente Gestão > Agência > Regional > Diretoria > Segmento
    }
}
```

### Prioridade de Filtros

A `FilterSpecification` aplica filtros seguindo uma hierarquia:

1. **Gerente** (mais específico)
2. **Gerente Gestão**
3. **Agência**
4. **Regional**
5. **Diretoria**
6. **Segmento** (menos específico)

Apenas o nível mais específico é aplicado.

## Operadores

### AndSpecification

Combina duas specifications com AND:

```php
$spec1 = new FilterSpecification($filters1);
$spec2 = new FilterSpecification($filters2);

$andSpec = new AndSpecification($spec1, $spec2);
// ou
$andSpec = $spec1->and($spec2);
```

### OrSpecification

Combina duas specifications com OR:

```php
$orSpec = $spec1->or($spec2);
```

### NotSpecification

Nega uma specification:

```php
$notSpec = $spec1->not();
```

## Exemplo Completo

```php
use App\Repository\Specification\FilterSpecification;
use App\Domain\DTO\FilterDTO;

// Criar filtros
$filters = new FilterDTO();
$filters->setSegmento('1');
$filters->setDiretoria('100');

// Criar specification
$spec = new FilterSpecification($filters);

// Aplicar em query
$qb = $this->getEntityManager()->createQueryBuilder();
$qb->select('e')
   ->from(DEstrutura::class, 'e');

$qb = $spec->apply($qb, 'e');

$results = $qb->getQuery()->getResult();
```

## Vantagens

1. **Flexibilidade**: Combina filtros de forma dinâmica
2. **Reutilização**: Specifications podem ser reutilizadas
3. **Testabilidade**: Fácil de testar isoladamente
4. **Manutenibilidade**: Lógica de filtros centralizada
5. **Composição**: Combina múltiplas condições facilmente

## Quando Usar

- **Use Specification Pattern quando:**
  - Filtros complexos e dinâmicos
  - Precisa combinar múltiplas condições
  - Queries que variam baseado em parâmetros

- **Não use quando:**
  - Filtros simples e fixos
  - Performance crítica (prefira SQL direto)
  - Queries muito específicas

---

**Anterior**: [Exceptions](./08-exceptions.md)  
**Próximo**: [Input Sanitizer](./10-security-input-sanitizer.md)  
**Voltar**: [Estrutura](./01-estrutura.md)
