# Backend - Security

## Autenticação via API Key

```php
<?php
namespace App\Security;

class ApiKeyAuthenticator extends AbstractGuardAuthenticator
{
    public function supports(Request $request): bool
    {
        // Não requer auth para documentação
        if (preg_match('#^/api/doc#', $request->getPathInfo())) {
            return false;
        }
        
        // Requer header X-API-Key
        return $request->headers->has('X-API-Key');
    }
    
    public function getCredentials(Request $request)
    {
        return $request->headers->get('X-API-Key');
    }
    
    public function getUser($credentials, UserProviderInterface $userProvider): ?UserInterface
    {
        $apiKey = $_ENV['API_KEY'] ?? $_SERVER['API_KEY'] ?? null;
        
        if (!$apiKey) {
            throw new CustomUserMessageAuthenticationException('API Key não configurada');
        }
        
        // Comparação segura (timing-safe)
        if (!hash_equals($apiKey, $credentials)) {
            throw new CustomUserMessageAuthenticationException('API Key inválida');
        }
        
        return new ApiKeyUser();
    }
}
```

## Configuração de Segurança

**config/packages/security.yaml**:
```yaml
security:
    providers:
        api_key_provider:
            id: App\Security\ApiKeyUserProvider
    
    firewalls:
        api:
            pattern: ^/api
            stateless: true
            guard:
                authenticators:
                    - App\Security\ApiKeyAuthenticator
    
    access_control:
        - { path: ^/api/(doc|health), roles: IS_AUTHENTICATED_ANONYMOUSLY }
        - { path: ^/api, roles: ROLE_API }
```

## Rotas Públicas

As seguintes rotas não requerem autenticação:
- `/api/doc` - Documentação Swagger
- `/api/doc.json` - Swagger JSON
- `/api/health` - Health check

## Input Sanitizer

```php
<?php
namespace App\Security;

class InputSanitizer
{
    public static function sanitize(string $input): string
    {
        // Remove tags HTML
        $input = strip_tags($input);
        
        // Remove caracteres de controle
        $input = preg_replace('/[\x00-\x1F\x7F]/', '', $input);
        
        // Trim
        $input = trim($input);
        
        return $input;
    }
}
```

---

**Próximo**: [Event Subscribers](./07-event-subscribers.md)  
**Ver também**: [Input Sanitizer](./10-security-input-sanitizer.md)
