# Backend - Repositories

Repositories são responsáveis pelo **acesso a dados**. Podem usar Doctrine ORM ou SQL direto.

## Exemplo com SQL Direto

```php
<?php
namespace App\Repository\Pobj;

use App\Domain\DTO\FilterDTO;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;

class ResumoRepository extends ServiceEntityRepository
{
    public function findProdutos(?FilterDTO $filters = null): array
    {
        $params = [];
        $types = [];
        $metaFilter = '';
        $realizadosFilter = '';
        
        // 1. Constrói filtros dinamicamente
        if ($filters) {
            $dataInicio = $filters->getDataInicio();
            if ($dataInicio) {
                $metaFilter .= " AND m.data_meta >= :metaDataInicio";
                $params['metaDataInicio'] = $dataInicio;
            }
            
            // Filtros de estrutura hierárquica
            $estruturaFilters = $this->buildEstruturaFilters($filters, $params, 'e1', 'meta');
            $metaFilter .= $estruturaFilters;
        }
        
        // 2. Constrói subqueries
        $metaSubquery = $this->buildMetaSubquery(
            $fMetaTable, 
            $dEstruturaTable, 
            $metaFilter, 
            $filters
        );
        
        // 3. Query principal com JOINs
        $sql = "SELECT 
                    dp.id,
                    dp.familia_id as id_familia,
                    f.nm_familia as familia,
                    COALESCE(fm.total_meta, 0) AS meta,
                    COALESCE(fr.total_realizado, 0) AS realizado
                FROM {$dProdutosTable} AS dp
                LEFT JOIN {$familiaTable} AS f ON f.id = dp.familia_id
                LEFT JOIN ({$metaSubquery}) AS fm ON fm.produto_id = dp.id
                LEFT JOIN ({$realizadosSubquery}) AS fr ON fr.produto_id = dp.id
                WHERE 1=1 {$produtoFilter}
                ORDER BY f.nm_familia ASC";
        
        // 4. Executa query
        $connection = $this->getEntityManager()->getConnection();
        $result = $connection->executeQuery($sql, $params, $types);
        
        // 5. Processa resultados
        $rows = [];
        while ($row = $result->fetchAssociative()) {
            $rows[] = $row;
        }
        $result->free();
        
        return $rows;
    }
    
    private function buildEstruturaFilters(
        ?FilterDTO $filters, 
        array &$params, 
        string $prefix = 'e'
    ): string {
        if (!$filters) {
            return '';
        }
        
        // Prioridade: Gerente > Gerente Gestão > Agência > Regional > Diretoria > Segmento
        if ($filters->getGerente()) {
            return ''; // Filtro aplicado diretamente
        } elseif ($filters->getAgencia()) {
            $key = "agenciaId";
            $params[$key] = $filters->getAgencia();
            return " AND {$prefix}.agencia_id = :{$key}";
        } elseif ($filters->getRegional()) {
            $key = "regionalId";
            $params[$key] = $filters->getRegional();
            return " AND {$prefix}.regional_id = :{$key}";
        }
        // ... outros níveis
        
        return '';
    }
}
```

## Padrões

- Sempre usar **prepared statements** (parâmetros nomeados)
- Usar `Connection::PARAM_STR_ARRAY` para arrays
- Sempre fazer `free()` do resultado após uso
- Construir queries dinamicamente baseado em filtros
- Subqueries para agregações complexas
- Nunca concatenar valores diretamente na query (SQL Injection)

## Quando Usar SQL Direto vs Doctrine ORM

**Use SQL Direto quando:**
- Queries complexas com múltiplas agregações
- Performance crítica
- Subqueries complexas
- Queries que não mapeiam bem para ORM

**Use Doctrine ORM quando:**
- Operações CRUD simples
- Relacionamentos entre entidades
- Migrações e schema management

---

**Próximo**: [Entities e DTOs](./05-entities-dtos.md)  
**Ver também**: [Specification Pattern](./09-specification-pattern.md)
