# Backend - Use Cases

Use Cases contêm a **lógica de negócio** da aplicação. Eles orquestram repositórios e transformam dados.

## Estrutura de um Use Case

```php
<?php
namespace App\Application\UseCase\Pobj;

use App\Domain\DTO\FilterDTO;
use App\Repository\Pobj\ResumoRepository;

class ResumoUseCase
{
    private $resumoRepository;

    public function __construct(ResumoRepository $resumoRepository)
    {
        $this->resumoRepository = $resumoRepository;
    }

    public function handle(?FilterDTO $filters = null): array
    {
        // 1. Busca dados brutos do repositório
        $produtos = $this->resumoRepository->findProdutos($filters);
        $produtosMensais = $this->resumoRepository->findProdutosMensais($filters);
        $variavel = $this->resumoRepository->findVariavel($filters);
        $calendario = $this->resumoRepository->findCalendario();
        
        // 2. Processa e transforma dados
        $businessSnapshot = $this->buildBusinessSnapshot($calendario);
        
        // 3. Converte para DTOs
        $cards = array_map(function($produto) {
            return CardDTO::fromArray($produto)->toArray();
        }, $produtos);
        
        // 4. Retorna estrutura final
        return [
            'cards' => $cards,
            'classifiedCards' => $classifiedCards,
            'variableCard' => $variableCard,
            'businessSnapshot' => $businessSnapshotDTO->toArray(),
        ];
    }
    
    private function buildBusinessSnapshot(array $calendario): array
    {
        // Lógica de negócio específica
    }
}
```

## Padrões

- Use Cases são **stateless** (sem estado)
- Método principal sempre se chama `handle()`
- Recebem `FilterDTO` ou `null`
- Retornam arrays estruturados
- Métodos privados para lógica auxiliar
- Nunca acessam diretamente o banco de dados (sempre via Repository)

## Principais Use Cases

- `ResumoUseCase`: Agrega dados de produtos, metas, realizados e variáveis
- `ProdutoUseCase`: Busca e agrega dados de produtos
- `RankingUseCase`: Calcula rankings de performance
- `ExecUseCase`: Dados para visão executiva
- `SimuladorUseCase`: Simulação de cenários
- `AgentUseCase`: Processamento de perguntas com IA
- `OmegaTicketsUseCase`: Gestão de chamados Omega

---

**Próximo**: [Repositories](./04-repositories.md)
