# Backend - Controllers

## ControllerBase

Todos os controllers herdam de `ControllerBase` que fornece métodos auxiliares:

```php
<?php
namespace App\Controller;

abstract class ControllerBase
{
    // Retorna resposta de sucesso padronizada
    protected function success($data): JsonResponse
    
    // Retorna resposta de erro (deprecated - usar exceções)
    protected function error(string $message, int $status = 400): JsonResponse
    
    // Lança exceção de validação
    protected function throwValidationError(string $message, array $errors = []): void
    
    // Lança exceção de recurso não encontrado
    protected function throwNotFound(string $message = 'Recurso não encontrado'): void
    
    // Lança exceção de requisição inválida
    protected function throwBadRequest(string $message = 'Requisição inválida'): void
}
```

## Exemplo de Controller

```php
<?php
namespace App\Controller\Pobj;

use App\Application\UseCase\Pobj\ResumoUseCase;
use App\Controller\ControllerBase;
use App\Domain\DTO\FilterDTO;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;

class ResumoController extends ControllerBase
{
    private $resumoUseCase;

    public function __construct(ResumoUseCase $resumoUseCase)
    {
        $this->resumoUseCase = $resumoUseCase;
    }

    /**
     * @Route("/api/pobj/resumo", name="api_pobj_resumo", methods={"GET"})
     */
    public function handle(Request $request): JsonResponse
    {
        // 1. Extrai parâmetros da query string
        $filters = new FilterDTO($request->query->all());
        
        // 2. Chama use case
        $result = $this->resumoUseCase->handle($filters);
        
        // 3. Retorna resposta padronizada
        return $this->success($result);
    }
}
```

## Padrões

- Controllers são **finos** - apenas recebem request e delegam
- Toda lógica de negócio fica em Use Cases
- Sempre usar `FilterDTO` para parâmetros de query
- Sempre retornar via `success()` ou lançar exceções
- Nunca fazer queries diretas no controller

## Principais Controllers

### POBJ
- `ResumoController`: Resumo de produtos e indicadores
- `DetalhesController`: Dados detalhados de produtos
- `RankingController`: Rankings de performance
- `ExecController`: Visão executiva
- `ProdutosController`: Gestão de produtos
- `SimuladorController`: Simulação de cenários
- `AgentController`: Chatbot com IA (OpenAI)

### Omega
- `OmegaTicketsController`: Gestão de chamados
- `OmegaUsersController`: Gestão de usuários
- `OmegaStatusController`: Status de chamados
- `OmegaStructureController`: Estrutura organizacional

### Encantabra
- `EncantabraPainelController`: Painel de regulamentos

---

**Próximo**: [Use Cases](./03-use-cases.md)
